<?php
namespace YES4Trade\Model;

use ebayAPI;

class ebaycalls {
    private const API_URL_LIVE = 'https://api.ebay.com/';
    private const API_URL_SANDBOX =  'https://api.sandbox.ebay.com/';
    private const TRADING_API_URL_SANDBOX = 'https://api.sandbox.ebay.com/ws/api.dll';
    private const TRADING_API_URL_LIVE = 'https://api.ebay.com/ws/api.dll';
    private const COMPAT_LEVEL = 1415;
    private const OAUTH2_USERTOKEN_FILE = 'ebay-sdk-oauth.usertoken';
    //private const OAUTH2_REFRESHTOKEN_FILE = main::get_secure_path().DIRECTORY_SEPARATOR.'ebay-sdk-oauth.refreshtoken';
 
    var $sandbox, $api_url, $credentials, $trading_api_url, $debug, $usertoken_file;

    public static function getUsertokenFile(){
        return \main::get_secure_path().DIRECTORY_SEPARATOR.self::OAUTH2_USERTOKEN_FILE;
    }
    public function __construct(bool $debug = false){
        $this->sandbox = EBAY_TESTMODE == '1';
        $this->api_url = ($this->sandbox)?self::API_URL_SANDBOX : self::API_URL_LIVE;
        $this->trading_api_url = ($this->sandbox)?self::TRADING_API_URL_SANDBOX : self::TRADING_API_URL_LIVE;
        $env = ($this->sandbox)?'test':'prod';
        $this->credentials = self::get_credentials($env);
        $this->debug = $debug;
    }

    
    /**
     * @param string $seller
     * @param string $refreshToken
     * 
     * @return string
     */
    public function getAccessTokenFromRefreshToken(string $refreshToken, array $scopes = []){
        return $this->query(
            'POST',
            $this->api_url . 'identity/v1/oauth2/token',
            [
                'headers'=>[
                    'Authorization' => 'Basic ' . $this->credentials,
                    'Content-Type'  => 'application/x-www-form-urlencoded',
                ],
                'form_params'=> [
                    'grant_type'    => 'refresh_token',
                    'refresh_token' => $refreshToken,
                    'scope'         => implode(' ', $scopes), // Wichtig: Leerzeichen, kein Komma
			    ]
            ]
        );
    }

    public function getFulfillmentPolicy(string $accessToken, string $marketplaceId){
        return $this->query(
            'GET',
            $this->api_url . 'sell/account/v1/fulfillment_policy',
            [
                'headers'=>[
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type'  => 'application/json',
                ],
                'query'=> [
                    'marketplace_id'    => $marketplaceId
			    ]
            ]
        );
    }

    public function getAccessAndRefreshToken(string $consent){
        $runame = YES_UNIQUE_RUNAME;
        return $this->query(
            'POST',
            $this->api_url . 'identity/v1/oauth2/token',
            [
                'headers'=>[
                    'Authorization' => 'Basic ' . $this->credentials,
                    'Content-Type'  => 'application/x-www-form-urlencoded',
                ],
                'form_params'=>[
                    'grant_type'    => 'authorization_code',
                    'redirect_uri' => $runame,
                    'code'=>$consent,
                ]
            ]
        );
    }

    private static function buildEbayRequest(string $callName, string $bodyXml, string $token): string {
        switch($callName){
            case 'GetTokenStatus':
                return <<<XML1
<?xml version="1.0" encoding="utf-8"?>
<{$callName}Request xmlns="urn:ebay:apis:eBLBaseComponents">
<RequesterCredentials>
    <eBayAuthToken>{$token}</eBayAuthToken>
  </RequesterCredentials>
</{$callName}Request>
XML1;
                break;
        }
        return <<<XML
<?xml version="1.0" encoding="utf-8"?>
<{$callName}Request xmlns="urn:ebay:apis:eBLBaseComponents">
  <RequesterCredentials>
    <eBayAuthToken>{$token}</eBayAuthToken>
  </RequesterCredentials>
  {$bodyXml}
</{$callName}Request>
XML;
    }

    private static function isNumericArray(array $array): bool {
        return array_keys($array) === range(0, count($array) - 1);
    }

    public static function array2xml($data):string{
        $xml = '';
        foreach ($data as $key => $value) {
            if (is_int($key)) {
                // Indexbasierter Schlüssel: Der Wert muss wissen, wie es heißt (z. B. ['OutputSelector' => [...]])
                // Wird im übergeordneten Aufruf behandelt → hier ignorieren
                continue;
            }

            if (is_array($value)) {
                // Prüfen ob es ein numerisches Array ist (mehrere gleichartige Einträge)
                if (self::isNumericArray($value)) {
                    foreach ($value as $subValue) {
                        if (is_array($subValue)) {
                            $xml .= sprintf('<%s>%s</%s>', $key, self::array2xml($subValue), $key);
                        } else {
                            $xml .= sprintf('<%s>%s</%s>', $key, htmlspecialchars($subValue), $key);
                        }
                    }
                } else {
                    // Assoziatives Array → verschachteltes XML
                    $xml .= sprintf('<%s>%s</%s>', $key, self::array2xml($value), $key);
                }
            } else {
                $xml .= sprintf('<%s>%s</%s>', $key, htmlspecialchars($value), $key);
            }
        }
        /*
        foreach ($data as $key => $value) {
            // Wenn numerischer Key: keine eigenen Tags generieren, nur Inhalt einfügen
            if (is_int($key)) {
                // Rekursiv verarbeiten, aber keine zusätzlichen <0>, <1> usw. erzeugen
                $xml .= self::array2xml($value);
            } elseif (is_array($value)) {
                $xml .= sprintf('<%s>%s</%s>', $key, self::array2xml($value), $key);
            } else {
                $xml .= sprintf('<%s>%s</%s>', $key, htmlspecialchars($value), $key);
            }
        }*/
        return $xml;
    }

    public function tradingQuery(string $seller, string $callName, int $siteId = -1, array $additional_xml_parameters=[]):object{
        if($siteId == -1){
            $siteId = EBAY_SITE;
        }
		$client = new \GuzzleHttp\Client();
        $ini = self::get_ebay_config_ini();
        $env = ($this->sandbox) ? 'test' : 'prod';
		$devId = $ini['ebay-config']['dev-key-'.$env];
		$appId = $ini['ebay-config']['app-key-'.$env];
		$certId = $ini['ebay-config']['cert-id-'.$env];
        $token = self::get_auth_token_from_file($seller);
        $bodyXml = '';
        if(sizeOf($additional_xml_parameters)){
            $bodyXml .= self::array2xml($additional_xml_parameters);
        }
        $call_params = [
            'headers' => [
                'X-EBAY-API-COMPATIBILITY-LEVEL' => self::COMPAT_LEVEL,
                'X-EBAY-API-DEV-NAME'            => $devId,
                'X-EBAY-API-APP-NAME'            => $appId,
                'X-EBAY-API-CERT-NAME'           => $certId,
                'X-EBAY-API-CALL-NAME'           => $callName,
                'X-EBAY-API-SITEID'              => $siteId,
                'Content-Type'                   => 'text/xml'
            ],
            'body' => self::buildEbayRequest($callName, $bodyXml, $token)
        ];
        //die(self::buildEbayRequest($callName, $bodyXml, $token));
        if($this->debug){
            printf(
                "<div style=\"color:#666;\">Seller %s: Send Post Request to %s<pre><code>\n\n%s</code></pre></div>",
                $seller,
                $this->trading_api_url,
                htmlentities(print_r($call_params,true))
            );
        }
        $response = $client->request('POST', $this->trading_api_url, $call_params);        
        $responseXML = (string)$response->getBody();
        // Schritt 1: XML in Objekt umwandeln
        $xml = simplexml_load_string($responseXML);

        // Schritt 2: in JSON und dann Array umwandeln
        $obj = json_decode(json_encode($xml));
        //die(print_r($obj,true));
		return $obj;
    }

    public static function get_auth_token_from_file(string $seller):string{
        $seller_safe = preg_replace( '/[^a-z0-9]+/', '-', strtolower( $seller ) );
        $seller_filename = str_replace('ebay-sdk-oauth.usertoken','ebay-sdk-oauth.'.$seller_safe.'.usertoken',self::getUsertokenFile());
        if(!file_exists($seller_filename)){
            throw new \Exception('Seller '.$seller.' has no token file');
        }
        $raw = file_get_contents($seller_filename);
        $json = json_decode($raw);
        return $json->access_token;
    }

    public function query(string $method, string $url, array $headerAndBody ){
		$client = new \GuzzleHttp\Client();
        $response = $client->request($method, $url, $headerAndBody);
   		$body_raw = (string)$response->getBody();
		return json_decode($body_raw);
	}

    private static function get_ebay_config_ini():array{
        if(!file_exists(\main::get_secure_path().'EBAY_DEVELOPER_KEYS.ini')){
            throw new \Exception('Missing developer keys ini file');
        }
        return parse_ini_file(\main::get_secure_path().'EBAY_DEVELOPER_KEYS.ini',true);
    }

    private static function get_credentials(string $env):string{
        $ini = self::get_ebay_config_ini();
		$clientId = $ini['ebay-config']['app-key-'.$env];
		$clientSecret = $ini['ebay-config']['cert-id-'.$env];//'DEIN_CLIENT_SECRET';
        return base64_encode("$clientId:$clientSecret");
    }

    public function getRedirectURL(){
        $sandbox = (defined('EBAY_TESTMODE') and constant('EBAY_TESTMODE') =='1')?true:false;
        $ini_conf = \ebayApi::get_ebay_config_ini();
        $cfg = $ini_conf['ebay-config'];
        $env = ($sandbox)?'test':'prod';
        $devID = $cfg['dev-key-'.$env];
        $appID = $cfg['app-key-'.$env];
        $certID = $cfg['cert-id-'.$env];
        $ruName = urlencode(constant('YES_UNIQUE_RUNAME')); // z. B. MyApp123-MyApp123-SBX-abcde1234
        if($sandbox){
            $ruName = 'agri_consulting-agricons-dccb-4-dttjfcj';
            $url = 'https://auth.sandbox.ebay.com/oauth2/authorize?client_id=agricons-dccb-49b6-be03-a3f5d9cf923c&response_type=code&redirect_uri=agri_consulting-agricons-dccb-4-czumskkr&scope=https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fbuy.order.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fbuy.guest.order%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.marketing.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.marketing%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.inventory.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.inventory%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.account.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.account%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.fulfillment.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.fulfillment%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.analytics.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.marketplace.insights.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.catalog.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fbuy.shopping.cart%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fbuy.offer.auction%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.identity.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.identity.email.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.identity.phone.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.identity.address.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.identity.name.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.identity.status.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.finances%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.payment.dispute%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.item.draft%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.item%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.reputation%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.reputation.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.notification.subscription%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.notification.subscription.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.stores%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.stores.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.vero';
        }else{
            $url = 'https://auth.ebay.com/oauth2/authorize?client_id='.$appID
            .'&response_type=code&redirect_uri='.$ruName
            .'&scope=https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.marketing.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.marketing%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.inventory.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.inventory%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.account.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.account%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.fulfillment.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.fulfillment%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.analytics.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.finances%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.payment.dispute%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.identity.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.reputation%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.reputation.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.notification.subscription%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.notification.subscription.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.stores%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fsell.stores.readonly%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fscope%2Fsell.edelivery%20https%3A%2F%2Fapi.ebay.com%2Foauth%2Fapi_scope%2Fcommerce.vero';
        }
        return $url;
    }
}